/** @file   Gradient.cpp
 * @brief   Implementation of Gradient - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#include "Gradient.h"

namespace eng2d {

//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Default constructor. A gradient from black to white through grey.
 */
Gradient::Gradient() :
  m_begCol( 0, 0, 0 ),
  m_midCol( 128, 128, 128 ),
  m_endCol( 255, 255, 255 ),
  m_bias( 50 )
{
}



/** Constructor.
 */
Gradient::Gradient(Color c1, Color c2) :
  m_begCol( c1 ),
  m_midCol( 0,0,0 ),
  m_endCol( c2 ),
  m_bias( 50 )
{
  CVal midR = (c1.r() + c2.r()) / 2;
  CVal midG = (c1.g() + c2.g()) / 2;
  CVal midB = (c1.b() + c2.b()) / 2;
  m_midCol.setRGB(midR, midG, midB);
}



/** Constructor
 */
Gradient::Gradient(Color c1, Color c2, Color c3) :
  m_begCol( c1 ),
  m_midCol( c2 ),
  m_endCol( c3 ),
  m_bias( 50 )
{
}



/** Constructor
 */
Gradient::Gradient(Color c1, Color c2, Color c3, int bias) :
  m_begCol( c1 ),
  m_midCol( c2 ),
  m_endCol( c3 ),
  m_bias( bias )
{
  if (m_bias < 1) m_bias = 1;
  if (m_bias > 99) m_bias = 99;
}



/** Copy constructor
 */
Gradient::Gradient(const Gradient& rO)
{
  m_begCol = rO.m_begCol;
  m_midCol = rO.m_midCol;
  m_endCol = rO.m_endCol;
  m_bias = rO.m_bias;
}



/** Destructor.
 */
Gradient::~Gradient()
{
}



/** Assignment operator
 */
Gradient& Gradient::operator = (const Gradient& rO)
{
  if (this != &rO) {
    m_begCol = rO.m_begCol;
    m_midCol = rO.m_midCol;
    m_endCol = rO.m_endCol;
    m_bias = rO.m_bias;
  }
  return *this;
}





//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Sets the 'from color'
 */
void Gradient::setC1(Color c)
{
  m_begCol = c;
}



/** Sets the 'through color'
 */
void Gradient::setC2(Color c)
{
  m_midCol = c;
}



/** Sets the 'to color'
 */
void Gradient::setC3(Color c)
{
  m_endCol = c;
}



/** Sets the biasing point for the gradient.
 */
void Gradient::setBias(float bias)
{
  m_bias = bias;
  if (m_bias < 1) m_bias = 1;
  if (m_bias > 99) m_bias = 99;
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Returns a color from the gradient.
 */
Color Gradient::getColorAt(int pos, int rangeStart, int rangeEnd) const
{
  if (pos < rangeStart) {
    return m_begCol;
  }
  if (pos > rangeEnd) {
    return m_endCol;
  }
  
  double p = static_cast<double>(pos);
  double rs = static_cast<double>(rangeStart);
  double re = static_cast<double>(rangeEnd);
  double bias = static_cast<double>(m_bias);
  
  p -= rs;
  re -= rs;
  if (rs < 0) {
    rs = -rs;
  }
  rs = 0;  
  double posPercent = (p / re) * 100;
  
  Color tmpCol;
  if (posPercent < bias) {
    // Find the color from transition 'begCol' to 'midCol'
    double diffR = static_cast<double>(m_midCol.r() - m_begCol.r());
    double diffG = static_cast<double>(m_midCol.g() - m_begCol.g());
    double diffB = static_cast<double>(m_midCol.b() - m_begCol.b());
    double posInTrans = posPercent / bias;
    int stepR = static_cast<int>( posInTrans * diffR );
    int stepG = static_cast<int>( posInTrans * diffG );
    int stepB = static_cast<int>( posInTrans * diffB);
    tmpCol.r( m_begCol.r() + stepR );
    tmpCol.g( m_begCol.g() + stepG );
    tmpCol.b( m_begCol.b() + stepB );
  } else {
    // Find the color from transition 'midCol' to 'endCol'
    double otherRange = 100.0 - bias;
    
    double diffR = static_cast<double>(m_endCol.r() - m_midCol.r());
    double diffG = static_cast<double>(m_endCol.g() - m_midCol.g());
    double diffB = static_cast<double>(m_endCol.b() - m_midCol.b());
    double posInTrans = (posPercent-bias) / otherRange;
    int stepR = static_cast<int>( posInTrans * diffR );
    int stepG = static_cast<int>( posInTrans * diffG );
    int stepB = static_cast<int>( posInTrans * diffB);
    tmpCol.r( m_midCol.r() + stepR );
    tmpCol.g( m_midCol.g() + stepG );
    tmpCol.b( m_midCol.b() + stepB );
  }
  return tmpCol;
}



/** Calculates the color from gradient
 */
void Gradient::color( float pos,float lowValue, float highValue, Color* pC ) const
{
  if ( pos < lowValue ) {
    pos = lowValue;
  } else if ( pos > highValue ) {
    pos = highValue;
  }
  
  // Translate the values so that we begin from 0
  pos -= lowValue;
  highValue -= lowValue;
  lowValue = 0;
  
  float posPercent = ( pos / highValue ) * 100;
  if ( posPercent < m_bias ) {
    Color diffCol = m_midCol - m_begCol;
    float posInTrans = posPercent / m_bias;
    float stepR = posInTrans * diffCol.r();
    float stepG = posInTrans * diffCol.g();
    float stepB = posInTrans * diffCol.b();
    pC->m_r = m_begCol.r() + stepR;
    pC->m_g = m_begCol.g() + stepG;
    pC->m_b = m_begCol.b() + stepB;
    
  } else {
    float otherRange = 100.0 - m_bias;
    Color diffCol = m_endCol - m_midCol;
    float posInTrans = ( posPercent - m_bias ) / otherRange;
    float stepR = posInTrans * diffCol.r();
    float stepG = posInTrans * diffCol.g();
    float stepB = posInTrans * diffCol.b();
    pC->m_r = m_midCol.r() + stepR;
    pC->m_g = m_midCol.g() + stepG;
    pC->m_b = m_midCol.b() + stepB;
  }
}


}
